function [fig, ax] = engMapOperation(eng, prof, contour_type, powerflows, yax, normalized)
arguments
    eng struct
    prof struct
    contour_type = "fc"
    powerflows string {mustBeText, mustBeMember(powerflows, ["all", "parallel", "series", "pe", "ps", "pt", "cd", "cs", "bc", "none", "bubbles"])}  = "none"
    yax string {mustBeMember(yax, ["power", "torque"])} = "torque"
    normalized = false
end
%engMapPlot
% Plot the engine operating points.
%
% Input arguments
% ---------------
% eng : struct
%   Engine data structure.
% prof : struct
%   Time profiles data structure.
% contour_type : string, optional
%   Specify: "fc" to plot the fuel flow rate map;
%            "bsfc" (default) for brake specific fuel consumption;
%            "eff" for fuel conversion efficiency.
% powerflows : string, optional
%   Specify one or more powerflows to represent in a string array. Specify:
%       "pe"  for pure electric (generator off);
%       "cd"  for charge-depleting;
%       "cs"  for charge-sustaining;
%       "bc"  for battery charging;
%       "all" for all powerflows.
%   The default is ["cd", "cs", "bc"].
%
% Outputs
% ---------------
% fig : Figure
%   Figure handle of the plot.
% ax : Axes
%   Axes handle of the plot.


%% Plot engine map
if isempty(contour_type)
    contour_type = "fc";
end
if isempty(powerflows)
    powerflows = "all";
end

[fig, ax] = engMapPlot(eng, contour_type, yax, normalized);

% Some compatibilty operations
if isfield(prof, "vehLoadFactor")
    pwrFlwCvName = "\lambda_{gen}";
    pwrFlwPrf = prof.vehLoadFactor;
elseif isfield(prof, "engTrqSplit")
    pwrFlwCvName = "\alpha_{eng}";
    pwrFlwPrf = prof.engTrqSplit;
else
    pwrFlwCvName = "missing";
    pwrFlwPrf = nan(size([prof.engSpd]));
end

if ~isfield(prof, "vehPwrFlw")
    vehPwrFlw = repmat("none", size([prof.engSpd]));
end

switch yax
    case "torque"
        yProf = [prof.engTrq];
    case "power"
        yProf = [prof.engSpd] .* [prof.engTrq] .* 1e-3; % kW;
end

%% Operating points scatter plot
if any(strcmp(powerflows, 'all'))
    powerflows = [ "pe", "ps", "pt", "cd", "cs", "bc"];
end

for n = 1:length(powerflows)

    powerflow = powerflows(n);
    idx = ismember(vehPwrFlw, powerflow);
    switch powerflow
        case 'pe'
            powerflow = "Pure electric";
            color = 'g';
        case 'ps'
            powerflow = "Power-split";
            color = 'b';
        case 'pt'
            powerflow = "Pure thermal";
            color = 'r';
        case 'none'
            powerflow = "Operating points";
            color = 'r';
        case 'cd'
            powerflow = "Charge depleting";
            color = 'b';
        case 'cs'
            powerflow = "Charge sustaining";
            color = 'r';
        case 'bc'
            powerflow = "Battery charging";
            color = 'k';
        case 'bubbles'
            color = "#1f77b4";
        otherwise
            continue
    end

    if strcmp(powerflow, "bubbles")
        idx = 1:length([prof.engSpd]);
        s = scatter([prof.engSpd] .* 30/pi, yProf, 100, 'o', ...
            'MarkerEdgeColor', "flat", 'MarkerEdgeAlpha', 0.1, ...
            'MarkerFaceColor', color, ...
            'MarkerFaceAlpha', 0.1, ...
            'DisplayName', "Operating points");
    else
        s = scatter([prof(idx).engSpd] .* 30/pi, yProf(idx), color, 'x', 'DisplayName', powerflow);
    end

    % Custom datatip
    s.DataTipTemplate.DataTipRows(1).Label = 'Speed';
    switch yax
        case "torque"
            s.DataTipTemplate.DataTipRows(2).Label = 'Torque';
        case "power"
            s.DataTipTemplate.DataTipRows(2).Label = 'Power';
    end
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow('Time, s', [prof(idx).time]);
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow('Gear Number', [prof(idx).gearNumber]);
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow(pwrFlwCvName, pwrFlwPrf(idx), '%.2f');
end

end